import { type Language } from "../api";
import { JsonObject } from "../common/json";
/**
 * Internalization provider, responsible for managing localizations and translating resources.
 */
export declare class I18nProvider {
    private readonly language;
    private readonly readTranslations;
    /**
     * Default language to be used when a resource does not exist for the desired language.
     */
    private static readonly DEFAULT_LANGUAGE;
    /**
     * Map of localized resources, indexed by their language.
     */
    private readonly _translations;
    /**
     * Initializes a new instance of the {@link I18nProvider} class.
     * @param language The default language to be used when retrieving translations for a given key.
     * @param readTranslations Function responsible for loading translations.
     */
    constructor(language: Language, readTranslations: TranslationsReader);
    /**
     * Translates the specified {@link key}, as defined within the resources for the {@link language}. When the key is not found, the default language is checked.
     *
     * Alias of `I18nProvider.translate(string, Language)`
     * @param key Key of the translation.
     * @param language Optional language to get the translation for; otherwise the default language.
     * @returns The translation; otherwise the key.
     */
    t(key: string, language?: Language): string;
    /**
     * Translates the specified {@link key}, as defined within the resources for the {@link language}. When the key is not found, the default language is checked.
     * @param key Key of the translation.
     * @param language Optional language to get the translation for; otherwise the default language.
     * @returns The translation; otherwise the key.
     */
    translate(key: string, language?: Language): string;
    /**
     * Gets the translations for the specified language.
     * @param language Language whose translations are being retrieved.
     * @returns The translations, otherwise `null`.
     */
    private getTranslations;
}
/**
 * Function responsible for providing localized resources.
 * @param language The language whose resources should be retrieved.
 * @returns Localized resources represented as a JSON object.
 */
export type TranslationsReader = (language: Language) => JsonObject | null;
/**
 * Parses the localizations from the specified contents, or throws a `TypeError` when unsuccessful.
 * @param contents Contents that represent the stringified JSON containing the localizations.
 * @returns The localizations; otherwise a `TypeError`.
 */
export declare function parseLocalizations(contents: string): JsonObject;
