import type { Coordinates, SetImage, SetTitle, State, WillAppear } from "../../api";
import type { JsonObject } from "../../common/json";
import type { KeyOf } from "../../common/utils";
import { Action } from "./action";
/**
 * Provides a contextualized instance of a key action.
 * @template T The type of settings associated with the action.
 */
export declare class KeyAction<T extends JsonObject = JsonObject> extends Action<T> {
    #private;
    /**
     * Initializes a new instance of the {@see KeyAction} class.
     * @param source Source of the action.
     */
    constructor(source: WillAppear<JsonObject>);
    /**
     * Coordinates of the key; otherwise `undefined` when the action is part of a multi-action.
     * @returns The coordinates.
     */
    get coordinates(): Coordinates | undefined;
    /**
     * Determines whether the key is part of a multi-action.
     * @returns `true` when in a multi-action; otherwise `false`.
     */
    isInMultiAction(): boolean;
    /**
     * Sets the {@link image} to be display for this action instance.
     *
     * NB: The image can only be set by the plugin when the the user has not specified a custom image.
     * @param image Image to display; this can be either a path to a local file within the plugin's folder, a base64 encoded `string` with the mime type declared (e.g. PNG, JPEG, etc.),
     * or an SVG `string`. When `undefined`, the image from the manifest will be used.
     * @param options Additional options that define where and how the image should be rendered.
     * @returns `Promise` resolved when the request to set the {@link image} has been sent to Stream Deck.
     */
    setImage(image?: string, options?: ImageOptions): Promise<void>;
    /**
     * Sets the current {@link state} of this action instance; only applies to actions that have multiple states defined within the manifest.
     * @param state State to set; this be either 0, or 1.
     * @returns `Promise` resolved when the request to set the state of an action instance has been sent to Stream Deck.
     */
    setState(state: State): Promise<void>;
    /**
     * Sets the {@link title} displayed for this action instance.
     *
     * NB: The title can only be set by the plugin when the the user has not specified a custom title.
     * @param title Title to display; when `undefined` the title within the manifest will be used.
     * @param options Additional options that define where and how the title should be rendered.
     * @returns `Promise` resolved when the request to set the {@link title} has been sent to Stream Deck.
     */
    setTitle(title?: string, options?: TitleOptions): Promise<void>;
    /**
     * Temporarily shows an "OK" (i.e. success), in the form of a check-mark in a green circle, on this action instance. Used to provide visual feedback when an action successfully
     * executed.
     * @returns `Promise` resolved when the request to show an "OK" has been sent to Stream Deck.
     */
    showOk(): Promise<void>;
    /**
     * @inheritdoc
     */
    toJSON(): object;
}
/**
 * Options that define how to render an image associated with an action.
 */
export type ImageOptions = Omit<KeyOf<SetImage, "payload">, "image">;
/**
 * Options that define how to render a title associated with an action.
 */
export type TitleOptions = Omit<KeyOf<SetTitle, "payload">, "title">;
