import type { PluginCommand, PluginEventMap, RegistrationInfo } from "../api";
import { EventEmitter } from "../common/event-emitter";
import { Version } from "./common/version";
/**
 * Provides a connection between the plugin and the Stream Deck allowing for messages to be sent and received.
 */
declare class Connection extends EventEmitter<ExtendedEventMap> {
    /**
     * Private backing field for {@link Connection.registrationParameters}.
     */
    private _registrationParameters;
    /**
     * Private backing field for {@link Connection.version}.
     */
    private _version;
    /**
     * Used to ensure {@link Connection.connect} is invoked as a singleton; `false` when a connection is occurring or established.
     */
    private canConnect;
    /**
     * Underlying web socket connection.
     */
    private connection;
    /**
     * Logger scoped to the connection.
     */
    private readonly logger;
    /**
     * Underlying connection information provided to the plugin to establish a connection with Stream Deck.
     * @returns The registration parameters.
     */
    get registrationParameters(): RegistrationParameters;
    /**
     * Version of Stream Deck this instance is connected to.
     * @returns The version.
     */
    get version(): Version;
    /**
     * Establishes a connection with the Stream Deck, allowing for the plugin to send and receive messages.
     * @returns A promise that is resolved when a connection has been established.
     */
    connect(): Promise<void>;
    /**
     * Sends the commands to the Stream Deck, once the connection has been established and registered.
     * @param command Command being sent.
     * @returns `Promise` resolved when the command is sent to Stream Deck.
     */
    send(command: PluginCommand): Promise<void>;
    /**
     * Gets the registration parameters, provided by Stream Deck, that provide information to the plugin, including how to establish a connection.
     * @returns Parsed registration parameters.
     */
    private getRegistrationParameters;
    /**
     * Attempts to emit the {@link ev} that was received from the {@link Connection.connection}.
     * @param ev Event message data received from Stream Deck.
     */
    private tryEmit;
}
/**
 * Registration information supplied by the Stream Deck when launching the plugin, that enables the plugin to establish a secure connection with the Stream Deck.
 */
type RegistrationParameters = {
    /**
     * Object containing information about the Stream Deck, this plugin, the user's operating system, user's Stream Deck devices, etc.
     */
    info: RegistrationInfo;
    /**
     * Unique identifier assigned to the plugin by Stream Deck; this value is used in conjunction with specific commands used to identify the source of the request, e.g. "getGlobalSettings".
     */
    pluginUUID: string;
    /**
     * Port used by the web socket responsible for communicating messages between the plugin and the Stream Deck.
     */
    port: string;
    /**
     * Name of the event used as part of the registration procedure when a connection with the Stream Deck is being established.
     */
    registerEvent: string;
};
/**
 * An extended event map that includes connection events.
 */
type ExtendedEventMap = PluginEventMap & {
    /**
     * Occurs when a connection is established.
     */
    connected: [info: RegistrationInfo];
};
export declare const connection: Connection;
export {};
