import type { ActionIdentifier, DeviceIdentifier } from "../../api";
import type { JsonValue } from "../../common/json";
import { type MessageGateway, type MessageRequestOptions, type MessageResponse } from "../../common/messaging";
import type { Action } from "../actions/action";
import type { DialAction } from "../actions/dial";
import type { KeyAction } from "../actions/key";
/**
 * Property inspector providing information about its context, and functions for sending and fetching messages.
 */
export declare class PropertyInspector implements Pick<MessageGateway<Action>, "fetch"> {
    private readonly router;
    /**
     * Action associated with the property inspector
     */
    readonly action: DialAction | KeyAction;
    /**
     * Initializes a new instance of the {@link PropertyInspector} class.
     * @param router Router responsible for fetching requests.
     * @param source Source the property inspector is associated with.
     */
    constructor(router: MessageGateway<Action>, source: ActionIdentifier & DeviceIdentifier);
    /**
     * Sends a fetch request to the property inspector; the property inspector can listen for requests by registering routes.
     * @template T The type of the response body.
     * @param request The request being sent to the property inspector.
     * @returns The response.
     * @example
     * // Within the property inspector, setup the route.
     * streamDeck.plugin.registerRoute("/show-dialog", () => {
     *   showDialog();
     * });
     *
     * // Within the plugin, send a fetch request to the current property inspector.
     * streamDeck.ui.current.fetch({
     *   path: "/show-dialog"
     * });
     */
    fetch<T extends JsonValue = JsonValue>(request: MessageRequestOptions): Promise<MessageResponse<T>>;
    /**
     * Sends a fetch request to the property inspector; the property inspector can listen for requests by registering routes.
     * @template T The type of the response body.
     * @param path Path of the request being sent to the property inspector.
     * @param body Optional body sent with the request.
     * @returns The response.
     * @example
     * // Within the property inspector, setup the route.
     * streamDeck.plugin.registerRoute("/show-dialog", () => {
     *   showDialog();
     * });
     *
     * // Within the plugin, send a fetch request to the current property inspector.
     * streamDeck.ui.current.fetch("/show-dialog");
     */
    fetch<T extends JsonValue = JsonValue>(path: string, body?: JsonValue): Promise<MessageResponse<T>>;
    /**
     * Sends the {@link payload} to the property inspector. The plugin can also receive information from the property inspector via {@link streamDeck.ui.onSendToPlugin} and {@link SingletonAction.onSendToPlugin}
     * allowing for bi-directional communication.
     * @template T The type of the payload received from the property inspector.
     * @param payload Payload to send to the property inspector.
     * @returns `Promise` resolved when {@link payload} has been sent to the property inspector.
     */
    sendToPropertyInspector(payload: JsonValue): Promise<void>;
}
